/***************************************************************
* Recode Raw Kansas Sentencing Commission Data
* Authors: Sanford Gordon and Gregory Huber w/ help from Nicole Simonelli                   
* * Update: February 27, 2007                                          
* Uses: Raw Kansas sentencing data                                
* New file created:
* "QJPS_activekssentences.dta" for data analysis
***************************************************************/

clear
set mem 200m
#delimit;
set more off;
capture log close;

/* THE FOLLOWING PATH STATEMENT IS SPECIFIC TO SANDY'S COMPUTER */
cd "C:\Documents and Settings\Sanford Gordon\My Documents\research\primary\criminal justice\Trial Judges\Kansas\coding";
log using "log files\recode sentencing data.log", replace;

/* Call the do file that takes annual sentencing datasets and appends the files into tbl_sentences_all.dta */
do "do files\call\Compress and append data.do";

use tbl_sentences_all.dta;

/* In Stata time, regime 1, which goes from July 1, 1995 to June 30, 1996 is 12965-13330
*                 regime 2, which goes from July 1, 1996 to June 30, 1999 is 13331-14425
*                 regime 3, which goes from July 1, 1999 to present, is 14426 onward 
* We want to code the regime under which the offense occurred                              */

/* For some reason, the primary offense date is formatted differently than the other offense dates */
/* Note: Throughout, 0 refers to the primary offense, and 1,2,3, etc. to additional offenses */

gen regime0 = 3;
replace regime0 = 2 if mdy(month(offdate),day(offdate),year(offdate)) <= 14425;
replace regime0 = 1 if mdy(month(offdate),day(offdate),year(offdate)) <=13330;
replace regime0 = . if mdy(month(offdate),day(offdate),year(offdate)) == .;
label variable regime0 "Guideline regime for top count";

gen regime1 = 3;
replace regime1 = 2 if mdy(month(offdate1),day(offdate1),year(offdate1)) <= 14425;
replace regime1 = 1 if mdy(month(offdate1),day(offdate1),year(offdate1)) <= 13330;
replace regime1 = . if mdy(month(offdate1),day(offdate1),year(offdate1)) == .;
label variable regime1 "Guideline regime for first additional count";

gen regime2 = 3;
replace regime2 = 2 if mdy(month(offdate2),day(offdate2),year(offdate2)) <= 14425;
replace regime2 = 1 if mdy(month(offdate2),day(offdate2),year(offdate2)) <= 13330;
replace regime2 = . if mdy(month(offdate2),day(offdate2),year(offdate2)) == .;
label variable regime2 "Guideline regime for second additional count";

gen regime3 = 3;
replace regime3 = 2 if mdy(month(offdate3),day(offdate3),year(offdate3)) <= 14425;
replace regime3 = 1 if mdy(month(offdate3),day(offdate3),year(offdate3)) <= 13330;
replace regime3 = . if mdy(month(offdate3),day(offdate3),year(offdate3)) == .;
label variable regime3 "Guideline regime for third additional count";

gen regime4 = 3;
replace regime4 = 2 if mdy(month(offdate4),day(offdate4),year(offdate4)) <= 14425;
replace regime4 = 1 if mdy(month(offdate4),day(offdate4),year(offdate4)) <= 13330;
replace regime4 = . if mdy(month(offdate4),day(offdate4),year(offdate4)) == .;
label variable regime4 "Guideline regime for fourth additional count";

gen regime5 = 3;
replace regime5 = 2 if mdy(month(offdate5),day(offdate5),year(offdate5)) <= 14425;
replace regime5 = 1 if mdy(month(offdate5),day(offdate5),year(offdate5)) <=13330;
replace regime5 = . if mdy(month(offdate5),day(offdate5),year(offdate5)) == .;
label variable regime5 "Guideline regime for fifth additional count";

/* Call the do files that code guideline sentences */
do "do files\call\topcount.do";
do "do files\call\addcount.do";
do "do files\call\bboxprobandlabel.do";

/*Number of additional counts in conviction */
gen addcount = (regime1!=.)+(regime2!=.)+(regime3!=.)+(regime4!=.)+(regime5!=.);

/* Procedural Characteristics of the Trial */
gen appcouns = counsel==1;
replace appcouns= . if counsel == .;
label variable appcouns "Appointed counsel";

gen pleabarg = trial == 3;
replace pleabarg = . if trial == .;
label variable pleabarg "Plea bargain (baseline is jury)";

gen benchtrial = trial == 1;
replace benchtrial = . if trial == .;
label variable benchtrial "Bench trial (baseline is jury)";

gen counts = adtotal;
replace counts = 0 if counts == .;
replace counts = counts+1;
label variable counts "Number of counts in the conviction";

/* Crime characteristics */

gen conspir = purpose == 2;
label variable conspir "Conspiracy";
gen solicit = purpose == 3;
label variable solicit "Solicitation";
gen attempt = purpose == 1;

/*  
** 1.  There are special rules that map into the guidelines, so we don't want to control for all of them 
** 2.  Make sure to get statutory authority associated with each DONE
** 3. NOTE: After the Kansas Supreme Court struck down the guidelines in 2001, those special rules associated 
**     with sentencing enhancements are factual determinations that must be made by a jury. 
** 4. NOTE ALSO: The KSC changed the special rule coding in 2000. 
*/

gen firearm = specrule == 1;
replace firearm = 1 if speother == "firearm"|speother == "used firearm"|speother == "with firearm";
label variable firearm "person felony committed with a firearm";

gen agbatleo = specrule == 2;
replace agbatleo = 1 if speother =="bat leo";
label variable agbatleo "aggravated battery on law enforcement official";

gen agassleo = specrule == 3;
replace agassleo = 1 if speother =="ag leo"|speother =="aggrav assault on leo"|speother =="aggravated assault LEO"|speother =="aslt LEO"|
speother =="aslt leo"|speother =="ass leo";
label variable agassleo "aggravated assault on law enforcement official";

gen strtgang = specrule == 4;
label variable strtgang "crime committed for benefit of criminal street gang";

gen pso = specrule == 5 & datayr >= 2000;
replace pso = 1 if speother == "sex off"|speother =="sex offe"|speother == "sexual o";
label variable pso "persistent sex offender";

gen fdui = specrule == 6 & datayr >= 2000;
replace fdui = 1 if specrule == 5 & datayr <=1999;
replace fdui = 1 if speother == "DUI"|speother=="Fel DUI"|speother=="dui"|speother=="fel dui"|speother == "felony dui";
label variable fdui "felony driving under the influence";

gen fcdmv = specrule == 7 & datayr >= 2000;
replace fcdmv = 1 if specrule == 6 & datayr <=1999;
label variable fcdmv "Felony criminal deprivation of a motor vehicle pre 7/1/99";

gen fdombat = specrule == 8 & datayr >= 2000;
replace fdombat = 1 if specrule == 7 & datayr <=1999;
replace fdombat = 1 if speother == "dom batr";
label variable fdombat "felony domestic battery";

gen probpar = specrule == 9 & datayr >= 2000;
replace probpar = 1 if specrule == 8 & datayr <=1999;
replace probpar = 1 if speother =="committed while in doc"|speother=="crime com while on felony dive"|speother=="crime on probation"|speother==
"doc"|speother=="doc cust"|speother =="felony probation"|speother=="in priso"|speother=="inc/felo"|speother=="on  probation"|speother=="on bprob"|
speother=="on parol"|speother=="on parole"|speother=="on prob"|speother=="on proba"|speother=="on probation"|speother=="parole"|speother=="prob & felony bond"|
speother=="prob,fel"|speother=="probatio"|speother=="probation"|speother=="probatn";
label variable probpar "crime committed while on probation or parole";

gen bond = specrule == 10 & datayr >= 2000;
replace bond = 1 if speother == "bond"|speother == "f.bond"|speother == "felony bond"|speother == "on bond"|speother == "on felony bond"|
speother == "on fln b";
label variable bond "crime commited while on felony bond";

gen exjuv = specrule == 11 & datayr >=2000;
replace exjuv = 1 if speother=="ext jv";
label variable exjuv "extended juvenile jurisdiction imposed";

gen mancnsb2 = specrule == 12 & datayr >=2000;
label variable mancnsb2 "second/subsequent manufacture of a controlled substance conv";

gen resbrgpr = specrule == 13 & datayr >= 2000;
replace resbrgpr = 1 if speother == "burg"|speother == "burglary"|speother == "res burg";
label variable resbrgpr "resident burglary with a prior resid,nonresid or agg burgl c";

gen othrule = specrule == 14 & datayr >= 2000;
replace othrule = 1 if specrule == 9 & datayr <= 1999;
label variable othrule "Other special sentencing rule";

save "QJPS_activekssentences.dta", replace;
clear;

/* Get data on defendant race, sex, birthdate */

/* Call the do file that takes annual demographic datasets and appends them into ksdemogdata1998to2003.dta */
do "Demographics\appenddemographicdata.do";
clear;
use "QJPS_activekssentences.dta";
sort id;
joinby id using "Demographics\ksdemogdata1998to2003.dta", unmatched(master);
drop _merge;
shell erase "Demographics\ksdemogdata1998to2003.dta";

/* Generate Demographic variables*/
gen nonwhite = (race !=1);
replace nonwhite = . if race == .;
label variable nonwhite "Defendant Nonwhite";

gen hispan = (ethnic == 1);
replace hispan = . if hispan == .;
label variable hispan "Defendant Hispanic";

gen male = (gender == 1);
label variable male "Defendant Male=1, female = 0";

/*Defendant AGE */

gen esent = mdy(month(sentdate),day(sentdate),year(sentdate));
gen ebirth = mdy(month(birthdat),day(birthdat),year(birthdat));
replace ebirth = mdy(month(birthdat),day(birthdat),1912) if year(ebirth) == 9012;
gen eage = esent-ebirth;
gen age = eage/365.25;
replace age = . if age < 15;
label variable age "Defendant age, in years";
gen age2 = age^2;
label variable age2 "Defendant age in years, squared";
drop ebirth eage race gender birthdat ethnic;

drop datayr ori_ kbi_ case_ criobjet ifyes courtrul fel_mis;

drop if esent < mdy(7,1,1997);
gen year = year(esent);
tab year, gen(y_);


gen grid1 = regime0 == 1;
gen grid2 = regime0 == 2;

/* Code victim characteristics for first crime */
gen v_govoff = chapter == 21 & (section == 3411|section==3413|section==3415);
label variable v_govoff "Victim was government official";
label variable v_govoff "Victim was government or law enforcement official";
gen v_child = (chapter == 21) & (section >= 3503 & section <=3506)|preoffen == 66;
label variable v_child "Victim was a child";
/* Code crime characteristics */
gen c_dead = (chapter == 21) & (section >=3401 & section <=3404) ;
gen c_assault = (chapter == 21) & section >=3410 & section <=3415 & section !=3412;
gen c_crimthreat = (chapter == 21) & section == 3419;
gen c_robbery = (chapter == 21) & (section == 3426|section==3427);
gen c_sex = (chapter == 21) & ((section >=3502 & section <= 3506)|section==3518);
gen c_theft = (chapter == 21) & (section == 3701|section==3704);
gen c_burglary = (chapter == 21) & (section == 3715|section==3716);
gen c_arson = (chapter == 21) & (section == 3718|section==3719);
egen touse = rsum(c_dead-c_arson);
drop if touse == 0;
drop touse;

/* Code total prison time including lower counts */
gen p0 = prisonmo;
replace p0 = 0 if p0 == .;
gen p1 = adpmo1;
replace p1 = 0 if p1 == .;
gen p2 = adpmo2;
replace p2 = 0 if p2 == .;
gen p3 = adpmo3;
replace p3 = 0 if p3 == .;
gen p4 = adpmo4;
replace p4 = 0 if p4 == .;
gen p5 = adpmo5;
replace p5 = 0 if p5 == .;
gen totpris = p0+(adcccs1==2)*p1+(adcccs2==2)*p2+(adcccs3==2)*p3+(adcccs4==2)*p4+(adcccs5==2)*p5;
label variable totpris "Total prison time including lower counts";
drop p0-p5;
gen lntotpris = ln(totpris+1);

/* Calculate judicial discretion 
*  Before State vs. Gould, an upward departure could go to twice the guideline maximum or 
*  four times the guideline maximum for the highest count in a multi-count conviction. 
*  Between 5/25/2001 and 6/6/2002, upward departures were struck down, so the guideline maximum
*  was the statutory maximum (twice the guideline maximum for the highest count in a multi-count 
*  conviction. After 6/6/2002, facts must be presented to a jury, but it is still the judge's 
*  discretion.
*/

gen statmax = 2*gmax0;
replace statmax = 4*gmax0 if addcount > 0;
replace statmax = gmax0 if addcount == 0 & sentdate > mdy(5,25,2001) & sentdate < mdy(6,6,2002);
replace statmax = 2*gmax0 if addcount == 1 & sentdate > mdy(5,25,2001) & sentdate < mdy(6,6,2002);
replace statmax = ln(1+statmax);

gen lntotprisl = lntotpris;
replace lntotprisl = . if lntotpris == 0;
replace lntotprisl = statmax if lntotpris > statmax;

gen lntotprisr = lntotpris;
replace lntotprisr = . if lntotpris > statmax;

/* Create separate variables for presumptive prison and presumptive probation on top counts */
gen gpres0_pris = (1-prob0)*(1-bbox0)*gpres0;
label variable gpres0_pris "Presumptive prison for top count";
gen gpres0_prob = gpres0-gpres0_pris;
label variable gpres0_prob "Presumptive probation for top count";

gen ln_gpres0_pris = ln(gpres0_pris+1);
gen ln_gpres0_prob = ln(gpres0_prob+1);
/*Drop the following judges, who have very few cases and for whom we can't verify when they were appointed:*/
rename judge judgeid;
drop if judgeid == 204;	
drop if judgeid == 272;	
drop if judgeid == 260;	
drop if judgeid == 207;	
drop if judgeid == 181;	
drop if judgeid == 192;	
drop if judgeid == 183;	
drop if judgeid == 185;	
drop if judgeid == 190;	
drop if judgeid == 213;	
drop if judgeid == 271;	
drop if judgeid == 264;	
drop if judgeid == 276;	
drop if judgeid == 244;	
drop if judgeid == 198;	
drop if judgeid == 200;	
drop if judgeid == 205;	
drop if judgeid == 218;	
drop if judgeid == 293;	
drop if judgeid == 246;	
drop if judgeid == 197;	
drop if judgeid == 280;	
drop if judgeid == 286;	
drop if judgeid == 235;	
drop if judgeid == 287;	
drop if judgeid == 267;	
drop if judgeid == 277;	
drop if judgeid == 236;	
drop if judgeid == 189;	
drop if judgeid == 191;	
drop if judgeid == 193;	
drop if judgeid == 186;	
drop if judgeid == 273;	
drop if judgeid == 188;	
drop if judgeid == 238;	
drop if judgeid == 290;	
drop if judgeid == 240;	
drop if judgeid == 177;	
drop if judgeid == 217;	
drop if judgeid == 233;	
drop if judgeid == 216;	
drop if judgeid == 195;	
drop if judgeid == 171;	
drop if judgeid == 75;
drop if judgeid == 234;
drop if judgeid == 283;
drop if judgeid == 178;
drop if judgeid == 43;
drop if judgeid == 175;
drop if judgeid == 74;
drop if judgeid == 3;
drop if judgeid == 245;
drop if judgeid == 284;
drop if judgeid == 50;
drop if judgeid == 215;
drop if judgeid == 281;
drop if judgeid == 27;
drop if judgeid == 112;
drop if judgeid == 279;
drop if judgeid == 25;
drop if judgeid == 77;
drop if judgeid == 184;
drop if judgeid == 194;
drop if judgeid == 232;
drop if judgeid == 47;
drop if judgeid == 248;
drop if judgeid == 151;
drop if judgeid == 5;
drop if judgeid == 7;
drop if judgeid == 52;
drop if judgeid == 59;
drop if judgeid == 71;
drop if judgeid == 84;
drop if judgeid == 88;
drop if judgeid == 106;
drop if judgeid == 126;
drop if judgeid == 199;
drop if judgeid == 214;
drop if judgeid == 222;
drop if judgeid == 227;
drop if judgeid == 288;
drop if judgeid == 176; /* This is Earle Jones -- he was elected in 1977; then retired. We omit him */
drop if judgeid == 261; /* This is clearly an appointed judge; appointed in 2001; didn't run in 2002. We omit. */

/*Get the district and division the presiding judge is serving in */
sort judgeid;

count;

joinby judgeid using "judges/judge_district.dta", unmatched(master);

count;

label variable district "Judicial district of judge";
label variable lntotpris "Natural log of total prison + 1";
label variable statmax "Statutory maximum sentence";
label variable lntotprisl "Left lntotpris for intreg";
label variable lntotprisr "Right lntotpris for intreg";
label variable ln_gpres0_pris "Log of 1+presumptive top count prison sentence";
label variable ln_gpres0_pris "Log of 1+presumptive top count probation sentence";
label variable addcount "Number of additional counts in conviction";
label variable attempt "Attempt";
label variable grid1 "First guideline regime";
label variable grid2 "Second guideline regime";
label variable c_dead "Crime involving a body";
label variable c_assault "Assault";
label variable c_crimthreat "Criminal Threat";
label variable c_robbery "Robbery";
label variable c_sex "Sex crime";
label variable c_theft "Theft";
label variable c_burglary "Burglary";
label variable c_arson "Arson";
label variable year "Year of sentence";
drop esent;

save "QJPS_activekssentences.dta", replace;

/*Erase the appended raw sentence data*/
shell erase "tbl_sentences_all.dta";
clear;
/**************************************************************************************
** SENTENCING RULES                                                                                         **
**************************************************************************************/

/** I. Upward Durational departures
**  KSA 21-4716 through 4718 provide for durational departures. Departures within 
**  the guidelines are not considered departures.
**  In State vs. Gould (2001) the Kansas Supreme Court declared portions of 
**  21-4718 unconstitutional on their face. The Kansas State Assembly amended the 
**  guidelines to be effective on June 6, 2002. Now, any aggravating factors other than
**  prior record that might enhance the maximum sentence must be submitted before a jury
**  and determined on a reasonable doubt basis. Note that the COURT determines whether
**  such a hearing (called a bifurcated jury proceeding) will be held. Thus, the decision on whether
**  to consider an upward departure appears to rest with the judge (i.e. the judge is an agenda setter).
**  For downward departures, the statute provides a nonexclusive list of mitigating factors that "can"
**  be taken into account.
** 21-4719(b)(2): "The presumptive term of imprisonment set in such departure shall not total more 
** than double the maximum duration of the presumptive imprisonment term."
** The judge can order (given appropriate determination of fact) a dispositional departure as well.
** FYI: In the data, most of the departure sentences are downward durational and dispositional.
** Only a few hundered are upward.
** 
** II. Anticipatory crimes -- see p. 29 of the Guidelines. Attempt, conspiracy, and solicitation are 
** treated as altering the severity level. For example, attempt lowers the severity level by 2 from
** the underlying/completed crime (no lower than 10). However, note that "selvel" ought to already 
** incorporate this.
**
** III. Special Rules
** Firearm: Presumed imprisonment
** Aggravated assault/battery on LEO: Presumed imprisonment
** Benefit of a criminal street gang: presumed imprisonment
** Persistent sex offender: Sentence for crimes with presumptive imprisonment shall be double the maximum duration of the 
** presumptive imprisonment term.
** Felony DUI: Non grid crime; offender cannot be sent to a state correctional facility. Amended in 2001 so that fourth or subsequent
** conviction can send you to KDOC.
** Felony domestic battery: Non-grid; sentence determined by statute. Offender cannot be sent to state correctional facility. 
** New felony while incarcerated/probation/community corrections/parole/conditional release/postrelease supervision: Court
** can sentence to prison even if new crime does not constitute departure
** Bond: ##Effective 7/1/99##, same rule as above.
** Kansas Securities Act:
** Extended Juvenile Jurisdiction
** Second/subsequent manufacture of controlled substance: Drug severity 1 and sentencing court must double presumptive sentence. 
** Sentencing court may reduce sentence in an amount not to excced 50% of special sentence length. 
** Burglary with prior: presumed imprisonment.
** Second forgery: non drug severity 8, at least 30 days imprisonment; Third and subsequent: at least 45 days imprisonment.
** 
** IV. Multiple convictions
** KSA 21-4720: Establishes that subject to the mandatory consecutive sentencing in 21-4608, the sentencing judge shall have discretion
** regarding whether to to sentence consecutively or concurrently. If the court is silent, it shall be presumed concurrent (21-4608(a)). 
** (N.B.This is reflected in the coding above.)
** subsection (c)(3): The total imprisonment term of the consecutive sentences, including the imprisonment term for the departure crime, 
    shall not exceed twice the maximum presumptive imprisonment term of the departure sentence following aggravation
** KSA 21-4608 mandatory consecutive sentences: THIS IS A POINT OF CONFUSION, BECAUSE THE STATUTE APPEARS TO SUGGEST THAT
** IN CASES OF, E.G, PAROLE, THE SENTENCE MUST BE CONSECUTIVE TO THE ONE THE PERSON IS ALREADY SERVING TIME FOR; BUT
** THE REFERENCE MANUAL SEEMS TO SUGGEST THAT IT MEANS MULTIPLE COUNT CONVICTIONS GIVEN THIS SITUATION MUST BE 
** SENTENCED CONSECUTIVELY.
** In multiple count cases, the total controlling sentence cannot exceed twice the base sentence (Note that this is the middle sentence on the grid, 
** not the maximum on the grid -- which applies in the case of upward duratinoal departures. KSA-4720(b)(4). This is an adjustment made
** after calculating the sentences for each count. However, if there are multiple convictions AND an upward departure, the double-double
** rule applies for the maximum. So if the top count has a grid-max of 100, and there are multiple counts, the absolute maximum is 400
** if the court wants the sentences to run consecutively.
** 
** 
*/
log close;
